# 🔌 API Odoo Integration

> Sistema de integração REST com Odoo ERP para gerenciamento de contatos, empresas, vínculos e automação de emails

[![PHP Version](https://img.shields.io/badge/PHP-%3E%3D7.4-blue)](https://www.php.net/)
[![License](https://img.shields.io/badge/license-MIT-green)](LICENSE)

## 📑 Índice

- [Visão Geral](#-visão-geral)
- [Estrutura do Projeto](#-estrutura-do-projeto)
- [Instalação](#-instalação)
- [Configuração](#-configuração)
- [APIs Disponíveis](#-apis-disponíveis)
  - [API Principal - Odoo](#1-api-principal---odoo)
  - [API Webhook - Emails](#2-api-webhook---emails)
  - [API Database - MySQL Direto](#3-api-database---mysql-direto)
- [Exemplos de Uso](#-exemplos-de-uso)
- [Arquitetura](#-arquitetura)
- [Desenvolvimento](#-desenvolvimento)
- [Logs e Monitoramento](#-logs-e-monitoramento)
- [Segurança](#-segurança)

---

## 🎯 Visão Geral

Este projeto fornece três APIs RESTful independentes:

1. **API Principal (`/public/`)** - Integração completa com Odoo ERP
2. **API Webhook (`/webhook/`)** - Sistema de webhooks e automação de emails
3. **API Database (`/BD/`)** - Acesso direto ao banco MySQL

### Principais Funcionalidades

- ✅ **Autenticação** via Bearer Token em todas as APIs
- 🔍 **Busca avançada** de contatos por email, CNPJ ou campos customizados
- 📝 **CRUD completo** de contatos e empresas no Odoo
- 🔗 **Gerenciamento de vínculos** entre usuários e empresas
- 📧 **Envio automatizado** de emails via SMTP
- 📊 **Sistema de logs** detalhado e centralizado
- 🗄️ **Gestão de banco** com criação e importação de schemas

---

## 📁 Estrutura do Projeto

```
api_clone/
├── public/                   # 🌐 API REST principal (Odoo)
│   └── index.php             # Endpoint principal da API
│
├── webhook/                  # 🔔 Sistema de webhooks e emails
│   ├── index.php             # Endpoint webhook
│   ├── Config.ini            # Configurações SMTP e tokens
│   ├── libs/
│   │   └── phpmailer/        # Biblioteca PHPMailer
│   │       ├── PHPMailer.php
│   │       ├── SMTP.php
│   │       └── Exception.php
│   └── src/
│       ├── Config.php        # Carregador de configurações
│       ├── Logger.php        # Sistema de logs do webhook
│       ├── Mailer.php        # Classe de envio de emails
│       ├── Webhook.php       # Processador principal de webhooks
│       ├── WebhookResponse.php
│       └── Actions/
│           ├── SendEmailAction.php          # Action de envio de email
│           └── WebhookActionInterface.php   # Interface para actions
│
├── BD/                       # 🗄️ API de gerenciamento MySQL
│   ├── index.php             # Endpoint de criação de bancos
│   └── sql/                  # Arquivos SQL para importação
│       ├── Dump20250712_Back.sql
│       └── Model_Modificado.sql
│
├── src/                      # 📦 Classes principais (API Odoo)
│   ├── config.php            # Credenciais Odoo e API token
│   ├── OdooClient.php        # Cliente JSON-RPC para Odoo
│   ├── EmailService.php      # Lógica de negócio principal
│   ├── Logger.php            # Sistema de logs centralizado
│   └── WebhookNotifier.php   # Notificador de webhooks
│
├── logs/                     # 📋 Logs automáticos (gerados em runtime)
│   └── api-YYYY-MM-DD.log    # Logs diários
│
├── .htaccess                 # Configurações Apache
├── .gitignore                # Arquivos ignorados pelo Git
└── README.md                 # Este arquivo
```

---

## 🚀 Instalação

### Requisitos

- PHP >= 7.4
- Extensões PHP: `curl`, `json`, `mysqli`
- Acesso ao servidor Odoo
- Servidor SMTP (para webhooks)

### Passos

1. **Clone o repositório**
```bash
git clone https://github.com/seu-usuario/api_odoo.git
cd api_odoo
```

2. **Configure as permissões**
```bash
chmod -R 755 .
mkdir -p logs
chmod 777 logs
```

3. **Configure o servidor web** (Apache/Nginx) ou use o servidor embutido do PHP:
```bash
php -S localhost:8000 -t public/
```

---

## ⚙️ Configuração

### 1. Configurar API Principal (Odoo)

Edite `src/config.php`:

```php
<?php
return [
    'url' => 'https://seu-dominio.odoo.com/',
    'db' => 'nome_database',
    'login' => 'seu-email@empresa.com',
    'token' => 'seu_token_api_odoo',
    'api_token' => 'TOKEN_SECRETO_PARA_API'
];
```

### 2. Configurar Webhook (Email)

Edite `webhook/Config.ini`:

```ini
[smtp]
host = "smtp.office365.com"
username = "seu-email@empresa.com"
password = "sua_senha_smtp"
port = 587

[webhook]
token = "TOKEN_SECRETO_WEBHOOK"
```

### 3. Configurar API Database

Edite as credenciais em `BD/index.php`:

```php
$host = 'seu_servidor_mysql';
$usuario = 'seu_usuario';
$senha = 'sua_senha';
$TOKEN_ESPERADO = 'TOKEN_SECRETO_BD_API';
```

> ⚠️ **IMPORTANTE**: Nunca commite credenciais reais no Git. Use variáveis de ambiente em produção.

---

## 📡 APIs Disponíveis

## 1. API Principal - Odoo

**Base URL**: `/public/`  
**Autenticação**: Bearer Token (definido em `src/config.php`)

### Endpoints

#### 🔍 Buscar Contato por Email

Busca contatos tanto em `res.partner` quanto na tabela de vínculos.

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "email": "usuario@exemplo.com"
}
```

**Response:**
```json
{
  "status": "ok",
  "dados": [
    {
      "empresa_id": 123,
      "empresa_nome": "Empresa XYZ Ltda",
      "usuario_id": 456,
      "usuario_nome": "João Silva",
      "status": "Ativo",
      "email": "usuario@exemplo.com",
      "telefone": "(11) 98765-4321",
      "cnpj": "12.345.678/0001-90",
      "tipo": "company",
      "fonte": "res.partner"
    }
  ]
}
```

---

#### 📝 Criar ou Atualizar Contato

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "email": "novo@exemplo.com",
  "name": "Maria Santos",
  "phone": "(11) 99999-8888",
  "vat": "12345678000190",
  "company_type": "person"
}
```

**Response:**
```json
{
  "status": "ok",
  "dados": {
    "id": 789,
    "name": "Maria Santos",
    "email": "novo@exemplo.com"
  }
}
```

---

#### 🏢 Buscar Empresa por CNPJ

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "acao": "buscarPorCNPJ",
  "cnpj": "12345678000190"
}
```

**Response:**
```json
{
  "status": "ok",
  "dados": [
    {
      "id": 100,
      "name": "Empresa ABC Ltda",
      "vat": "12.345.678/0001-90",
      "email": "contato@empresaabc.com",
      "phone": "(11) 3333-4444",
      "company_type": "company"
    }
  ]
}
```

---

#### 🔗 Adicionar Vínculo Usuário-Empresa

Cria relação entre usuário (pessoa) e empresa na aba de vínculos do Odoo.

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "acao": "adicionar_usuario_aba",
  "empresa": "empresa@exemplo.com",
  "usuario": "usuario@exemplo.com",
  "nome": "João Silva",
  "status": "Ativo"
}
```

**Response:**
```json
{
  "status": "ok"
}
```

---

#### ✏️ Atualizar Vínculo

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "acao": "atualizar_usuario_aba",
  "empresa": "empresa@exemplo.com",
  "usuario": "usuario@exemplo.com",
  "status": "Inativo"
}
```

---

#### 🔎 Busca Genérica por Campo

Permite busca flexível em qualquer campo do Odoo.

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "campo": "phone",
  "valor": "11999998888",
  "fields": ["id", "name", "email", "phone"]
}
```

---

#### 🔗 Buscar Relações Many2Many

Para campos relacionais no Odoo.

**Request:**
```bash
POST /public/
Authorization: Bearer SEU_TOKEN_API
Content-Type: application/json

{
  "campo": "parent_id",
  "valor": 123,
  "manyField": "child_ids",
  "fields": ["id", "name", "email"]
}
```

---

## 2. API Webhook - Emails

**Base URL**: `/webhook/`  
**Autenticação**: Bearer Token (definido em `webhook/Config.ini`)

### Endpoints

#### 📧 Enviar Email

Envia emails via SMTP configurado (Office365, Gmail, etc).

**Request:**
```bash
POST /webhook/
Authorization: Bearer SEU_TOKEN_WEBHOOK
Content-Type: application/json

{
  "action": "send-email",
  "email-config": {
    "to": "destinatario@exemplo.com",
    "subject": "Assunto do Email",
    "body": "<h1>Olá!</h1><p>Este é um email HTML.</p>",
    "from": "remetente@empresa.com",
    "fromName": "Nome do Remetente"
  }
}
```

**Response (Sucesso):**
```json
{
  "success": true,
  "message": "Email enviado com sucesso"
}
```

**Response (Erro):**
```json
{
  "success": false,
  "message": "Falha ao enviar email: [detalhes do erro]"
}
```

### Campos Opcionais do Email

- `cc`: Cópias (string ou array)
- `bcc`: Cópias ocultas (string ou array)
- `replyTo`: Email para resposta
- `attachments`: Array de anexos (paths)

**Exemplo com anexos:**
```json
{
  "action": "send-email",
  "email-config": {
    "to": "cliente@exemplo.com",
    "subject": "Proposta Comercial",
    "body": "<p>Segue proposta em anexo.</p>",
    "attachments": [
      "/path/to/proposta.pdf",
      "/path/to/contrato.docx"
    ]
  }
}
```

---

## 3. API Database - MySQL Direto

**Base URL**: `/BD/`  
**Autenticação**: Bearer Token (definido em `BD/index.php`)

### Endpoint

#### 🗄️ Criar Banco e Importar SQL

Cria novo banco de dados MySQL e opcionalmente importa arquivos SQL.

**Request:**
```bash
POST /BD/
Authorization: Bearer SEU_TOKEN_BD
Content-Type: application/json

{
  "database": "novo_banco",
  "sql_file": "Model_Modificado.sql"
}
```

**Response (Sucesso):**
```json
{
  "status": "ok",
  "mensagem": "Banco 'novo_banco' criado com sucesso",
  "arquivos_executados": ["Model_Modificado.sql"]
}
```

**Response (Banco já existe):**
```json
{
  "erro": "Banco 'novo_banco' já existe."
}
```

### Executar SQL Inline

Você também pode enviar SQL diretamente no payload:

**Request:**
```bash
POST /BD/
Authorization: Bearer SEU_TOKEN_BD
Content-Type: application/json

{
  "database": "novo_banco",
  "sql_inline": "CREATE TABLE usuarios (id INT PRIMARY KEY, nome VARCHAR(100));"
}
```

---

## 💡 Exemplos de Uso

### Cenário 1: Cadastro Completo de Cliente

```bash
# 1. Criar empresa
curl -X POST https://api.empresa.com/public/ \
  -H "Authorization: Bearer MEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "email": "contato@clienteabc.com",
    "name": "Cliente ABC Ltda",
    "vat": "12345678000190",
    "phone": "(11) 3333-4444",
    "company_type": "company"
  }'

# 2. Criar usuário pessoa
curl -X POST https://api.empresa.com/public/ \
  -H "Authorization: Bearer MEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "email": "joao@clienteabc.com",
    "name": "João Silva",
    "phone": "(11) 99999-8888",
    "company_type": "person"
  }'

# 3. Vincular usuário à empresa
curl -X POST https://api.empresa.com/public/ \
  -H "Authorization: Bearer MEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "acao": "adicionar_usuario_aba",
    "empresa": "contato@clienteabc.com",
    "usuario": "joao@clienteabc.com",
    "nome": "João Silva",
    "status": "Ativo"
  }'

# 4. Enviar email de boas-vindas
curl -X POST https://api.empresa.com/webhook/ \
  -H "Authorization: Bearer TOKEN_WEBHOOK" \
  -H "Content-Type: application/json" \
  -d '{
    "action": "send-email",
    "email-config": {
      "to": "joao@clienteabc.com",
      "subject": "Bem-vindo ao sistema!",
      "body": "<h1>Olá João!</h1><p>Seu cadastro foi realizado com sucesso.</p>"
    }
  }'
```

---

### Cenário 2: Consulta e Atualização

```bash
# 1. Buscar dados do cliente
curl -X POST https://api.empresa.com/public/ \
  -H "Authorization: Bearer MEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"email": "cliente@exemplo.com"}'

# 2. Atualizar status do vínculo
curl -X POST https://api.empresa.com/public/ \
  -H "Authorization: Bearer MEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "acao": "atualizar_usuario_aba",
    "empresa": "empresa@exemplo.com",
    "usuario": "cliente@exemplo.com",
    "status": "Inativo"
  }'
```

---

## 🏗️ Arquitetura

### Classes Principais

#### `src/OdooClient.php`
Cliente HTTP para comunicação com API JSON-RPC do Odoo.

**Métodos principais:**
- `getUserId()`: Autentica e retorna ID do usuário
- `request($model, $method, $args, $fields)`: Requisição genérica ao Odoo

**Modelos Odoo utilizados:**
- `res.partner`: Contatos e empresas
- `x_vinculo_usuario_por_`: Vínculos usuário-empresa

---

#### `src/EmailService.php`
Lógica de negócio para operações no Odoo.

**Métodos principais:**
- `buscarPorEmail($email)`: Busca completa de contato
- `buscarIdPorEmail($email)`: Retorna apenas ID
- `buscarPorCNPJ($cnpj)`: Busca empresa por CNPJ
- `criarOuAtualizarContato($acao, $email, $dados)`: CRUD de contatos
- `adicionarOuAtualizarVinculoPlataformaPorEmail()`: Gerencia vínculos

---

#### `src/Logger.php`
Sistema centralizado de logs.

**Características:**
- Logs diários em `logs/api-YYYY-MM-DD.log`
- Timestamp automático
- Thread-safe (flock)

**Uso:**
```php
use Src\Logger;

Logger::log("Mensagem de log");
Logger::log("Erro: " . $exception->getMessage());
```

---

#### `webhook/src/Webhook.php`
Processador de webhooks com validações.

**Validações automáticas:**
- Token de autenticação
- Content-Type (application/json)
- Método HTTP (POST)
- Estrutura do payload

**Padrão Strategy** para actions:
```php
interface WebhookActionInterface {
    public function execute(array $data);
}
```

---

#### `webhook/src/Actions/SendEmailAction.php`
Action para envio de emails via PHPMailer.

**Configurações SMTP suportadas:**
- Office365
- Gmail
- SMTP customizado

---

### Fluxo de Dados

#### API Principal (Odoo)
```
Cliente HTTP
    ↓
Bearer Token Validation
    ↓
public/index.php (Router)
    ↓
EmailService (Business Logic)
    ↓
OdooClient (HTTP JSON-RPC)
    ↓
Odoo ERP
    ↓
Logger (Logs)
```

#### API Webhook
```
Sistema Externo
    ↓
Bearer Token Validation
    ↓
webhook/index.php
    ↓
Webhook::handle()
    ↓
SendEmailAction::execute()
    ↓
PHPMailer → SMTP
    ↓
Logger (Logs)
```

---

## 👨‍💻 Desenvolvimento

### Adicionar Nova Action no Webhook

1. **Criar classe da action:**

```php
// webhook/src/Actions/MinhaNovaAction.php
<?php

class MinhaNovaAction implements WebhookActionInterface {
    
    public function execute(array $data) {
        // Validar dados
        if (!isset($data['campo_obrigatorio'])) {
            throw new Exception("Campo obrigatório ausente");
        }
        
        // Processar lógica
        $resultado = $this->minhaLogica($data);
        
        // Retornar resposta
        return [
            'success' => true,
            'message' => 'Ação executada com sucesso',
            'data' => $resultado
        ];
    }
    
    private function minhaLogica($data) {
        // Implementação
    }
}
```

2. **Registrar no webhook/index.php:**

```php
require_once __DIR__ . '/src/Actions/MinhaNovaAction.php';

// No método handle() da classe Webhook:
$actions = [
    'send-email' => new SendEmailAction($this->logger),
    'minha-nova-action' => new MinhaNovaAction($this->logger)
];
```

---

### Estender EmailService

```php
// Em src/EmailService.php

public function meuNovoMetodo($parametro) {
    Logger::log("Executando meuNovoMetodo com: $parametro");
    
    try {
        $resultado = $this->odoo->request(
            'res.partner',
            'search_read',
            [['campo', '=', $parametro]],
            ['id', 'name', 'email']
        );
        
        Logger::log("Resultado: " . json_encode($resultado));
        return $resultado;
        
    } catch (\Exception $e) {
        Logger::log("Erro: " . $e->getMessage());
        throw $e;
    }
}
```

---

### Criar Nova API

1. Criar diretório e arquivo principal:
```bash
mkdir minha_api
touch minha_api/index.php
```

2. Implementar validação e lógica:
```php
<?php
header('Content-Type: application/json');

require_once '../src/Logger.php';
require_once '../src/OdooClient.php';

use Src\Logger;
use Src\OdooClient;

// Validar token
$config = require '../src/config.php';
$headers = getallheaders();
$authHeader = $headers['Authorization'] ?? '';

if (!preg_match('/Bearer\s(\S+)/', $authHeader, $matches) || 
    $matches[1] !== $config['api_token']) {
    http_response_code(401);
    echo json_encode(['error' => 'Token inválido']);
    exit;
}

// Processar requisição
$input = json_decode(file_get_contents('php://input'), true);

try {
    // Sua lógica aqui
    $resultado = minhaFuncao($input);
    echo json_encode(['status' => 'ok', 'dados' => $resultado]);
    Logger::log("API executada com sucesso");
    
} catch (Exception $e) {
    Logger::log("Erro: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Erro interno']);
}
```

---

## 📊 Logs e Monitoramento

### Estrutura dos Logs

Arquivos: `logs/api-YYYY-MM-DD.log`

**Formato:**
```
[2025-12-08 14:23:45] Mensagem de log
[2025-12-08 14:23:46] Consulta por e-mail realizada com sucesso
[2025-12-08 14:23:50] Erro ao processar requisição: Token inválido
```

### Visualizar Logs

```bash
# Logs em tempo real
tail -f logs/api-$(date +%Y-%m-%d).log

# Últimas 50 linhas
tail -n 50 logs/api-$(date +%Y-%m-%d).log

# Buscar por termo
grep "Erro" logs/api-$(date +%Y-%m-%d).log

# Logs do dia anterior
tail -f logs/api-$(date -d "yesterday" +%Y-%m-%d).log

# Todos os logs do mês
cat logs/api-2025-12-*.log | grep "sucesso"
```

### Rotação de Logs

Os logs são automaticamente separados por dia. Para limpeza periódica:

```bash
# Remover logs com mais de 30 dias
find logs/ -name "api-*.log" -mtime +30 -delete

# Compactar logs antigos
find logs/ -name "api-*.log" -mtime +7 -exec gzip {} \;
```

---

## 🔒 Segurança

### Checklist de Segurança

- [ ] **Tokens únicos e complexos** para cada ambiente
- [ ] **HTTPS obrigatório** em produção
- [ ] **Variáveis de ambiente** para credenciais
- [ ] **Validação de input** em todos os endpoints
- [ ] **Rate limiting** implementado
- [ ] **Logs sem dados sensíveis**
- [ ] **Firewall** configurado (whitelist de IPs)
- [ ] **Backups regulares** do banco de dados

### Usar Variáveis de Ambiente

**Recomendado para produção:**

```php
// src/config.php
<?php
return [
    'url' => getenv('ODOO_URL'),
    'db' => getenv('ODOO_DB'),
    'login' => getenv('ODOO_LOGIN'),
    'token' => getenv('ODOO_TOKEN'),
    'api_token' => getenv('API_TOKEN')
];
```

**Configurar no servidor:**
```bash
# Apache (.htaccess)
SetEnv ODOO_URL "https://seu-dominio.odoo.com/"
SetEnv ODOO_TOKEN "seu_token_aqui"

# Nginx
env ODOO_URL;
env ODOO_TOKEN;

# Docker
environment:
  - ODOO_URL=https://seu-dominio.odoo.com/
  - ODOO_TOKEN=seu_token_aqui
```

### Rate Limiting (Exemplo)

```php
// Adicionar no início de public/index.php
function checkRateLimit($identifier) {
    $file = "/tmp/ratelimit_$identifier";
    $limit = 100; // requisições por hora
    $window = 3600; // 1 hora em segundos
    
    if (file_exists($file)) {
        $data = json_decode(file_get_contents($file), true);
        if (time() - $data['start'] < $window) {
            if ($data['count'] >= $limit) {
                http_response_code(429);
                die(json_encode(['error' => 'Rate limit exceeded']));
            }
            $data['count']++;
        } else {
            $data = ['start' => time(), 'count' => 1];
        }
    } else {
        $data = ['start' => time(), 'count' => 1];
    }
    
    file_put_contents($file, json_encode($data));
}

// Usar IP como identificador
$ip = $_SERVER['REMOTE_ADDR'];
checkRateLimit($ip);
```

---

## 📝 Modelos Odoo

### res.partner

Modelo principal de contatos e empresas.

**Campos importantes:**
| Campo | Tipo | Descrição |
|-------|------|-----------|
| `id` | Integer | ID único do registro |
| `name` | String | Nome do contato/empresa |
| `email` | String | Email principal |
| `phone` | String | Telefone |
| `mobile` | String | Celular |
| `vat` | String | CNPJ/CPF |
| `company_type` | Selection | 'person' ou 'company' |
| `parent_id` | Many2one | Empresa pai (para contatos) |
| `child_ids` | One2many | Contatos vinculados |
| `street` | String | Endereço |
| `city` | String | Cidade |
| `state_id` | Many2one | Estado |
| `zip` | String | CEP |
| `country_id` | Many2one | País |

---

### x_vinculo_usuario_por_

Modelo customizado para vínculos usuário-empresa.

**Campos importantes:**
| Campo | Tipo | Descrição |
|-------|------|-----------|
| `x_studio_e_mail` | String | Email do usuário |
| `x_studio_empresa` | Many2one | Empresa vinculada |
| `x_studio_usuario_pessoa` | Many2one | Usuário pessoa (res.partner) |
| `x_studio_status` | Selection | Status: Ativo/Inativo |

---

## 🐛 Troubleshooting

### Erro: "Token inválido"

**Causa**: Token não configurado ou incorreto.

**Solução**:
1. Verificar `src/config.php` → `api_token`
2. Confirmar que o header está correto: `Authorization: Bearer SEU_TOKEN`

---

### Erro: "Erro de conexão com Odoo"

**Causa**: URL, database ou credenciais incorretas.

**Solução**:
1. Verificar `src/config.php`
2. Testar acesso manual ao Odoo
3. Verificar logs: `tail -f logs/api-*.log`

---

### Emails não sendo enviados

**Causa**: Configuração SMTP incorreta ou porta bloqueada.

**Solução**:
1. Verificar `webhook/Config.ini`
2. Testar conectividade: `telnet smtp.office365.com 587`
3. Verificar firewall/antivírus
4. Conferir logs do webhook

---

### Logs não sendo gerados

**Causa**: Permissões do diretório `logs/`.

**Solução**:
```bash
chmod 777 logs/
chown www-data:www-data logs/  # Linux
```

---

## 📦 Backups

### Backup Manual

```bash
# Backup do código
tar -czf backup-api-$(date +%Y%m%d).tar.gz \
  --exclude='logs/*' \
  --exclude='vendor/*' \
  .

# Backup do banco MySQL
mysqldump -h HOST -u USER -p DATABASE > backup-$(date +%Y%m%d).sql
```

### Backup Automatizado (Cron)

```bash
# Adicionar no crontab (crontab -e)
# Backup diário às 2h da manhã
0 2 * * * /home/user/scripts/backup-api.sh

# Script backup-api.sh:
#!/bin/bash
cd /var/www/api_clone
tar -czf /backups/api-$(date +\%Y\%m\%d).tar.gz .
find /backups -name "api-*.tar.gz" -mtime +30 -delete
```

---

## 🧪 Testes

### Testar Autenticação

```bash
# Token válido - deve retornar 400 (esperando dados)
curl -X POST http://localhost:8000/public/ \
  -H "Authorization: Bearer SEU_TOKEN" \
  -H "Content-Type: application/json"

# Token inválido - deve retornar 401
curl -X POST http://localhost:8000/public/ \
  -H "Authorization: Bearer TOKEN_ERRADO" \
  -H "Content-Type: application/json"

# Sem token - deve retornar 401
curl -X POST http://localhost:8000/public/ \
  -H "Content-Type: application/json"
```

### Testar Busca

```bash
# Busca por email existente
curl -X POST http://localhost:8000/public/ \
  -H "Authorization: Bearer SEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"email": "teste@exemplo.com"}'

# Busca por CNPJ
curl -X POST http://localhost:8000/public/ \
  -H "Authorization: Bearer SEU_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"acao": "buscarPorCNPJ", "cnpj": "12345678000190"}'
```

### Testar Webhook

```bash
# Envio de email simples
curl -X POST http://localhost:8000/webhook/ \
  -H "Authorization: Bearer TOKEN_WEBHOOK" \
  -H "Content-Type: application/json" \
  -d '{
    "action": "send-email",
    "email-config": {
      "to": "teste@exemplo.com",
      "subject": "Teste",
      "body": "Corpo do teste"
    }
  }'
```

---

## 📚 Recursos Adicionais

- [Documentação Odoo API](https://www.odoo.com/documentation/16.0/developer/reference/external_api.html)
- [PHPMailer GitHub](https://github.com/PHPMailer/PHPMailer)
- [JSON-RPC Specification](https://www.jsonrpc.org/specification)

---

## 📄 Licença

Este projeto é proprietário. Todos os direitos reservados.

---

## 👥 Contribuindo

Para contribuir com o projeto:

1. Crie um branch para sua feature: `git checkout -b feature/nova-funcionalidade`
2. Commit suas mudanças: `git commit -m 'Adiciona nova funcionalidade'`
3. Push para o branch: `git push origin feature/nova-funcionalidade`
4. Abra um Pull Request

---

## 📞 Suporte

Para questões técnicas ou dúvidas:
- Consulte os logs em `logs/`
- Verifique este README
- Entre em contato com a equipe de desenvolvimento

---

**Última atualização**: 8 de dezembro de 2025
