<?php

namespace Src;

class EmailService
{
    private $odoo;

    public function __construct(OdooClient $odoo)
    {
        $this->odoo = $odoo;
    }

    public function listarTodosCampos()
    {
        $dominio = [];

        try {
            $resultado = $this->odoo->request('res.partner', 'search_read', $dominio, []);
            if (!$resultado) {
                Logger::log("Nenhum dado encontrado.");
                return [];
            }

            Logger::log("Resultado da consulta de todos os campos: " . json_encode($resultado));
            foreach ($resultado as &$item) {
                Logger::log("Campos retornados: " . json_encode($item));
            }

            return $resultado;
        } catch (\Exception $e) {
            Logger::log("Erro ao consultar o Odoo: " . $e->getMessage());
            return [];
        }
    }

    public function buscarEmail($email, $extraFields = [])
    {
        Logger::log("🔍 Buscando e-mail em vínculos e em res.partner: $email");

        $modeloVinculo = $this->getModeloVinculoPlataforma() ?? 'x_vinculo_usuario_por_';
        $emailLimpo = strtolower(trim($email));

        $empresasComUsuario = [];

        try {
            // 1️⃣ Busca na aba de vínculos
            $usuarios = $this->odoo->request($modeloVinculo, 'search_read', [
                [['x_studio_e_mail', '=', $emailLimpo]]
            ], ['id', 'x_studio_empresa', 'x_studio_usuario_pessoa', 'x_studio_status', 'x_studio_e_mail']);

            Logger::log("📦 Vínculos encontrados: " . json_encode($usuarios));

            foreach ($usuarios as $usuario) {
                $empresasComUsuario[] = [
                    'empresa_id'   => $usuario['x_studio_empresa'][0] ?? null,
                    'empresa_nome' => $usuario['x_studio_empresa'][1] ?? 'Sem nome',
                    'usuario_id'   => $usuario['x_studio_usuario_pessoa'][0] ?? null,
                    'usuario_nome' => $usuario['x_studio_usuario_pessoa'][1] ?? 'Desconhecido',
                    'status'       => $usuario['x_studio_status'] ?? 'Desconhecido',
                    'email'        => $usuario['x_studio_e_mail'] ?? '',
                    'fonte'        => 'aba_vinculo'
                ];
            }

            // 2️⃣ Busca também em res.partner (campo principal "email")
            $partners = $this->odoo->request('res.partner', 'search_read', [
                [['email', '=', $emailLimpo]]
            ], array_merge(['id', 'name', 'email', 'phone', 'vat', 'company_type'], $extraFields));

            Logger::log("📦 Partners encontrados: " . json_encode($partners));

            foreach ($partners as $partner) {
                $empresasComUsuario[] = [
                    'empresa_id'   => $partner['id'] ?? null,
                    'empresa_nome' => $partner['name'] ?? 'Sem nome',
                    'usuario_id'   => null,
                    'usuario_nome' => null,
                    'status'       => 'Principal',
                    'email'        => $partner['email'] ?? '',
                    'telefone'     => $partner['phone'] ?? '',
                    'cnpj'         => $partner['vat'] ?? '',
                    'tipo'         => $partner['company_type'] ?? '',
                    'fonte'        => 'res.partner'
                ];
            }

            Logger::log("✅ Resultado final: " . json_encode($empresasComUsuario));
            return $empresasComUsuario;

        } catch (\Exception $e) {
            Logger::log("⚠️ Erro ao buscar e-mail: " . $e->getMessage());
            return [];
        }
    }


    public function buscarIdPorEmail($email)
{
    Logger::log("🔍 Buscando ID em res.partner por email exato: $email");

    try {
        $ids = $this->odoo->request('res.partner', 'search', [[['email', '=', strtolower(trim($email))]]]);

        if (!empty($ids)) {
            Logger::log("✅ ID encontrado: " . $ids[0]);
            return $ids[0]; // apenas o primeiro encontrado
        }

        Logger::log("⚠️ Nenhum ID encontrado em res.partner para o e-mail: $email");
        return null;

    } catch (\Throwable $th) {
        Logger::log("❌ Erro ao buscar ID por email no res.partner: " . $th->getMessage());
        return null;
    }
}



    public function buscarEmpresasPorId($ids)
{
    $modeloVinculo = $this->getModeloVinculoPlataforma();

    $empresas = $this->odoo->request('res.partner', 'search_read', [
        [['id', 'in', $ids]]
    ], ['id', 'name', 'vat', 'x_studio_produtos', 'x_studio_vinculos_plataforma_novo']);

    foreach ($empresas as &$empresa) {
        $empresa['empresa_nome'] = $empresa['name'] ?? 'Sem nome';

        $idsVinculos = $empresa['x_studio_vinculos_plataforma_novo'] ?? [];
        $idsVinculos = is_array($idsVinculos) ? $idsVinculos : [$idsVinculos];

        $usuarios = [];

        if (!empty($idsVinculos)) {
            $vinculos = $this->odoo->request(
                $modeloVinculo,
                'search_read',
                [[['id', 'in', $idsVinculos]]],
                ['id', 'x_name', 'x_email', 'x_usuario_pessoa', 'x_empresa', 'x_status']
            );

            foreach ($vinculos as $v) {
                $usuarios[] = [
                    'id'              => $v['id'],
                    'nome'            => $v['x_name'] ?? null,
                    'email'           => $v['x_email'] ?? null,
                    'usuario_pessoa'  => is_array($v['x_usuario_pessoa']) ? $v['x_usuario_pessoa'][1] : null,
                    'empresa_id'      => is_array($v['x_empresa']) ? $v['x_empresa'][0] : null,
                    'empresa_nome'    => is_array($v['x_empresa']) ? $v['x_empresa'][1] : null,
                    'status'          => $v['x_status'] ?? null, // ✅ Aqui também incluído no JSON final
                ];
            }
        }

        $empresa['usuarios_plataforma'] = $usuarios;

        if (!empty($empresa['x_studio_produtos'])) {
            $empresa['produtos'] = $this->buscarProdutosPorIds($empresa['x_studio_produtos']);
        }
    }

    return $empresas;
}



    public function buscarProdutosPorIds($produtosIds)
    {
        if (empty($produtosIds)) {
            Logger::log("Nenhum ID de produto foi fornecido para a busca.");
            return [];
        }

        $produtosIds = array_values(array_filter(array_map('intval', $produtosIds)));

        if (empty($produtosIds)) {
            Logger::log("Todos os IDs de produto eram inválidos.");
            return [];
        }

        Logger::log("Buscando todos os produtos para filtrar os IDs: " . json_encode($produtosIds));

        $defaultFields = ['id', 'name', 'list_price', 'description_sale', 'image_1920'];

        try {
            $todosProdutos = $this->odoo->request('product.template', 'search_read', [], $defaultFields);

            Logger::log("Total de produtos retornados: " . count($todosProdutos));

            $produtosFiltrados = array_filter($todosProdutos, function ($produto) use ($produtosIds) {
                return in_array($produto['id'], $produtosIds);
            });

            Logger::log("Produtos filtrados: " . json_encode(array_values($produtosFiltrados)));

            return array_values($produtosFiltrados);
        } catch (\Exception $e) {
            Logger::log("Erro ao buscar produtos: " . $e->getMessage());
            return [];
        }
    }

    public function buscarPorEmail($email)
{
    Logger::log("Iniciando busca por e-mail: $email");

    $vinculos = $this->buscarEmail($email);

    if (empty($vinculos)) {
        Logger::log("Nenhum vínculo encontrado para o e-mail: $email");
        return ["error" => "E-mail não encontrado em nenhum vínculo"];
    }

    $modeloVinculo = $this->getModeloVinculoPlataforma();
    $empresasFinal = [];

    foreach ($vinculos as $vinculo) {
        $empresaId = $vinculo['empresa_id'];
        $usuarioId = $vinculo['usuario_id'];
        $statusVinculo = $vinculo['status']; // ✅ Reaproveita status capturado
        $fonte = $vinculo['fonte'] ?? '';


        //if (!$empresaId || !$usuarioId) continue;
        if (!$empresaId) continue;
        if (!$usuarioId && $fonte !== 'res.partner') continue;


        $empresaDados = $this->odoo->request('res.partner', 'search_read', [
            [['id', '=', $empresaId]]
        ], ['id', 'name', 'vat', 'x_studio_produtos', 'x_studio_vinculos_plataforma_novo']);

        if (empty($empresaDados)) continue;

        $empresa = $empresaDados[0];
        $empresa['empresa_nome'] = $empresa['name'] ?? 'Sem nome';

        // Reaproveita os dados já buscados
        $empresa['usuarios_plataforma'] = [[
            'id'              => null,
            'nome'            => $vinculo['usuario_nome'],
            'email'           => $vinculo['email'],
            'usuario_pessoa'  => $vinculo['usuario_nome'],
            'empresa_id'      => $empresaId,
            'empresa_nome'    => $empresa['empresa_nome'],
            'status'          => $statusVinculo, // ✅ Aqui ele aparece corretamente
        ]];

        // Produtos (se houver)
        if (!empty($empresa['x_studio_produtos'])) {
            $empresa['produtos'] = $this->buscarProdutosPorIds($empresa['x_studio_produtos']);
        }

        $empresasFinal[] = $empresa;
    }

    Logger::log("📤 DEBUG FINAL: " . json_encode(["status" => "ok", "dados" => $empresasFinal]));
    return ["status" => "ok", "dados" => $empresasFinal];
}


    public function criarOuAtualizarContato($acao, $email, $dadosNovos)
{
    Logger::log("📨 Ação recebida: $acao para o e-mail: $email");

    if (!in_array($acao, ['criar', 'atualizar'])) {
        Logger::log("🚫 Ação inválida: $acao");
        return ["error" => "Ação inválida"];
    }

    if (empty($email)) {
        Logger::log("⚠️ E-mail ausente para ação de $acao");
        return ["error" => "E-mail obrigatório"];
    }

    if (!is_array($dadosNovos) || empty($dadosNovos)) {
        Logger::log("⚠️ Nenhum dado foi fornecido para ação de $acao");
        return ["error" => "Dados obrigatórios ausentes"];
    }

    // Converte lista de e-mails em IDs (para campo many2many se necessário)
    if (isset($dadosNovos['x_studio_usurios']) && is_array($dadosNovos['x_studio_usurios'])) {
        $dadosNovos['x_studio_usurios'] = $this->converterEmailsParaIds($dadosNovos['x_studio_usurios']);
    }

    if ($acao === 'atualizar') {
        $id = $this->buscarIdPorEmail($email); // <- agora busca direto no res.partner

        if (!$id) {
            Logger::log("🚫 Contato com e-mail $email não encontrado para atualização");
            return ["error" => "Contato com e-mail $email não encontrado"];
        }

        try {
            $this->odoo->request('res.partner', 'write', [[intval($id)], $dadosNovos]);
            Logger::log("✅ Contato atualizado: ID $id com dados: " . json_encode($dadosNovos));
            return ["status" => "Contato atualizado", "id" => $id];
        } catch (\Exception $e) {
            Logger::log("❌ Erro ao atualizar contato: " . $e->getMessage());
            return ["error" => "Erro ao atualizar: " . $e->getMessage()];
        }
    }

    if ($acao === 'criar') {
        $dadosNovos['email'] = $email;

        try {
            $idCriado = $this->odoo->request('res.partner', 'create', [$dadosNovos]);
            Logger::log("✅ Novo contato criado: ID $idCriado com dados: " . json_encode($dadosNovos));
            return ["status" => "Contato criado", "id" => $idCriado];
        } catch (\Exception $e) {
            Logger::log("❌ Erro ao criar contato: " . $e->getMessage());
            return ["error" => "Erro ao criar: " . $e->getMessage()];
        }
    }

    return ["error" => "Erro inesperado"];
}


    private function converterEmailsParaIds($emails)
    {
        $ids = [];

        foreach ($emails as $email) {
            $resultado = $this->buscarEmail($email, ['id']);
            if (!empty($resultado) && isset($resultado[0]['id'])) {
                $ids[] = $resultado[0]['id'];
            } else {
                Logger::log("Email $email não encontrado para conversão em ID.");
            }
        }

        return $ids;
    }

    public function getModeloVinculoPlataforma()
    {
        try {
            $modelos = $this->odoo->request('ir.model', 'search_read', [
                [['model', 'ilike', 'vinculo'], ['model', 'ilike', 'usuario']]
            ], ['model', 'name']);

            Logger::log('Modelos encontrados: ' . json_encode($modelos));

            foreach ($modelos as $m) {
                if (strpos($m['model'], 'vinculo') !== false && strpos($m['model'], 'usuario') !== false) {
                    return $m['model'];
                }
            }

            return null;
        } catch (\Exception $e) {
            Logger::log("Erro ao buscar modelo de vínculo: " . $e->getMessage());
            return null;
        }
    }

    public function adicionarOuAtualizarVinculoPlataformaPorEmail($empresa, $usuario, $nome, $status)
{
    Logger::log("📥 Adicionando ou atualizando vínculo com os dados:");
    Logger::log("Empresa: $empresa | Usuário: $usuario | Nome: $nome | Status: $status");

    $modeloVinculo = $this->getModeloVinculoPlataforma();

    // Verifica se empresa é ID numérico ou e-mail
    if (is_numeric($empresa)) {
        $empresaId = (int)$empresa;
        Logger::log("🏢 ID da empresa recebido diretamente: $empresaId");
    } else {
        $empresaId = $this->buscarIdEmpresaPorEmail($empresa);
        if (!$empresaId) {
            Logger::log("❌ Empresa não encontrada: $empresa");
            return false;
        }
        Logger::log("✅ Empresa encontrada: ID $empresaId");
    }

    // Busca ID do usuário
    $usuarioId = $this->buscarIdUsuarioPorEmail($usuario);
    if (!$usuarioId) {
        Logger::log("❌ Usuário não encontrado: $usuario");
        return false;
    }
    Logger::log("👤 Usuário encontrado: ID $usuarioId");

    // Verifica se já existe vínculo
    Logger::log("🔎 Verificando vínculos existentes para a empresa...");
    $vinculosEmpresa = $this->odoo->request($modeloVinculo, 'search_read', [
        [['x_studio_empresa', '=', $empresaId]]
    ], ['id']);


    $ehPrimeiro = empty($vinculosEmpresa);
    if ($ehPrimeiro) {
        Logger::log("🆕 Nenhum vínculo anterior encontrado — será marcado como Admin");
    }

    // Verifica se já existe vínculo
    Logger::log("🔎 Verificando vínculo existente...");
    $vinculos = $this->odoo->request($modeloVinculo, 'search_read', [[
        ['x_studio_empresa', '=', $empresaId],
        ['x_studio_usuario_pessoa', '=', $usuarioId]
    ], ['id']]);

        
    if (!empty($vinculos)) {
        $vinculoIdExistente = $vinculos[0]['id'];
        Logger::log("🔁 Vínculo já existe (ID $vinculoIdExistente), atualizando status");

        $resultado = $this->odoo->request(
            model: $modeloVinculo,
            method: 'write',
            args: [[$vinculoIdExistente], [
                'x_studio_status' => $status,
                'x_studio_nome' => $ehPrimeiro ? 'Admin' : $nome
            ]]
        );


        Logger::log("✅ Vínculo atualizado com sucesso: ID $vinculoIdExistente");
    } else {
        Logger::log("🔧 Criando novo vínculo");

        $resultado = $this->odoo->request($modeloVinculo, 'create', [[
            'x_studio_nome' => $ehPrimeiro ? 'Admin' : $nome,
            'x_studio_e_mail' => $usuario,
            'x_studio_empresa' => $empresaId,
            'x_studio_usuario_pessoa' => $usuarioId,
            'x_studio_status' => $status
        ]]);

        Logger::log("✅ Novo vínculo criado para usuário $usuario com empresa $empresa");
    }

    return $resultado;
}



    public function buscarIdEmpresaPorEmail($empresa)
{
    Logger::log("📨 Buscando empresa com valor recebido: " . $empresa);

    // Caso seja ID numérico
    if (is_numeric($empresa)) {
        Logger::log("🔢 Empresa informada como ID: " . $empresa);
        return (int)$empresa;
    }

    // Verifica se é um e-mail válido
    if (!filter_var($empresa, FILTER_VALIDATE_EMAIL)) {
        Logger::log("🛑 Valor de empresa inválido: " . $empresa);
        return false;
    }

    // Busca todas as empresas e filtra manualmente
    try {
        //$empresas = $this->odoo->request('res.partner', 'search_read', [], ['fields' => ['id', 'name', 'email']]);
        $empresas = $this->odoo->request('res.partner', 'search_read', [], ['id', 'name', 'email']);

        Logger::log("🔎 Total de empresas retornadas: " . count($empresas));

        foreach ($empresas as $e) {
            if (isset($e['email']) && strtolower(trim($e['email'])) === strtolower(trim($empresa))) {
                Logger::log("✅ Empresa encontrada: ID " . $e['id'] . " - " . $e['name']);
                return $e['id'];
            }
        }

        Logger::log("❌ Nenhuma empresa com e-mail correspondente encontrada.");
        return false;

    } catch (\Exception $e) {
        Logger::log("🛑 Erro ao buscar empresas: " . $e->getMessage());
        return false;
    }
}





public function buscarIdUsuarioPorEmail($email)
{
    Logger::log("🔍 Buscando ID do usuário com e-mail: $email");

    $resultado = $this->odoo->request('res.partner', 'search_read', [
        [['email', '=', $email]]
    ], ['id']);

    if (!empty($resultado)) {
        $id = $resultado[0]['id'];
        Logger::log("✅ ID do usuário encontrado: $id");
        return $id;
    }

    Logger::log("❌ Usuário não encontrado com o e-mail: $email");
    return false;
}


public function buscarPorCNPJ($cnpj)
{
    Logger::log("🔍 Buscando empresa com CNPJ : $cnpj");

    try {
        $res = $this->odoo->request('res.partner', 'search_read', [
            //[['vat', '=', $cnpj]]
            [['vat', 'ilike', $cnpj]]

        ], [
            'id','name','email','phone','vat',
            'street','street2','city','zip','state_id','website'
        ]);

      
        if (empty($res)) {
            Logger::log("❌ Nenhuma empresa encontrada com esse CNPJ : ".$cnpj);
            return ['status' => 'erro', 'mensagem' => 'Empresa não encontrada : '.$cnpj];
        }

        $p = $res[0];

        // --- Normaliza dados vindos do Odoo ---
        $dados = [
            'cnpj'        => $p['vat'] ?? '',
            'nome'        => $p['name'] ?? '',
            'email'       => $p['email'] ?? '',
            //'telefone'    => $p['phone'] ?? '',
            'telefone' => substr(preg_replace('/\D+/', '', $p['phone'] ?? ''), -11),
            'site'        => $p['website'] ?? '',
            'cep'         => $p['zip'] ?? '',
            'logradouro'  => $p['street'] ?? '',
            'complemento' => $p['street2'] ?? '',
            'cidade'      => $p['city'] ?? '',
            'uf'          => '',
            'numero'      => '', // se tiver campo específico no Odoo, preencha aqui
            'bairro'      => '', // idem
        ];

        // helper simples p/ SQL (mantém NULL quando valor vazio)
        $q = fn($v) => ($v === null || $v === '') ? 'NULL' : ("'".str_replace("'", "''", $v)."'");

        // caminhos
        $sqlDir       = __DIR__ . '/../BD/sql';
        if (!is_dir($sqlDir)) { mkdir($sqlDir, 0775, true); }
        $novoDumpPath = $sqlDir . '/Model_Modificado.sql';

        // carrega o modelo base (se quiser concatenar ao final)
        $dumpBase = __DIR__ . '/../BD/sql/Dump_lgpd_model.sql';
        $conteudoBase = file_exists($dumpBase) ? file_get_contents($dumpBase) : '';

        // bloco de atualizações
        $sqlAtualizacao = <<<SQL

SET FOREIGN_KEY_CHECKS=0;

TRUNCATE TABLE `LGPD_0001_tenant`;
TRUNCATE TABLE `LGPD_1000_controlador`;
TRUNCATE TABLE `LGPD_0002_conf_smpt`;
TRUNCATE TABLE `sec_users`;
TRUNCATE TABLE `sec_users_groups`;

/*
DELETE FROM `sec_groups_apps` WHERE `group_id` <> 1;
INSERT INTO `sec_groups_apps`
SELECT * FROM `lgpd_model`.`sec_groups_apps`
WHERE group_id = 1;
*/

INSERT INTO `LGPD_0001_tenant` (`cnpj_0001`)
VALUES ({$q($dados['cnpj'])});

INSERT INTO `LGPD_1000_controlador` (
  `nome_1000`,`CEP_1000`,`web_site_1000`,`logo_1000`,`telefone_1000`,
  `cnpj_1000`,`id_controlador_1000`,`id_tp_atua_contr_1000`,
  `id_area_atua_contr_1000`,`dpo_1000`,`atv_1000`,`logradouro_1000`,
  `bairro_1000`,`uf_1000`,`cidade_1000`,`número_logr_1000`,`compl_logr_1000`,
  `tipo_logr_1000`,`tipo_dpo_1000`,`dpo_externo_2090`,`resp_dpo_ext_1000`,
  `email_resp_dpo_ext_1000`,`tel_resp_dpo_ext_1000`
) VALUES (
  {$q($dados['nome'])}, {$q($dados['cep'])}, {$q($dados['site'])}, NULL, {$q($dados['telefone'])},
  {$q($dados['cnpj'])}, 1, 1,
  1, NULL, NULL, {$q($dados['logradouro'])},
  {$q($dados['bairro'])}, {$q($dados['uf'])}, {$q($dados['cidade'])}, {$q($dados['numero'])}, {$q($dados['complemento'])},
  {$q('Rua')}, NULL, NULL, NULL,
  NULL, NULL
);

INSERT INTO `LGPD_0002_conf_smpt` (
  `id_controlador_1000`,`url_0002`,`mail_smtp_server_0002`,`mail_smtp_user_0002`,
  `mail_smtp_pass_0002`,`mail_smtp_from_0002`,`url_portal_acesso_0002`,
  `url_recover_portal_titular_0002`,`url_register_portal_titular_0002`,`porta_smtp`
) VALUES (
  1,
  'http://168.75.92.221/OMN_PCP/conf_conn_app/',
  'smtp.office365.com',
  'plataforma@omnisblue.com',
  'Zaw14627',
  'plataforma@omnisblue.com',
  'http://168.75.92.221/OMN_ACS/conf_change_conne/',
  'http://168.75.92.221/OMN_TIT/control_change_password/',
  'http://168.75.92.221/OMN_TIT/conf_conn_confirm_email/',
  587
);

INSERT INTO `sec_users` (
  `login`,`pswd`,`name`,`email`,`active`,`activation_code`,`priv_admin`,`mfa`,
  `picture`,`cpf`,`telefone`,`controlador`,`cargo`,`organograma`,`nome`,`id_user`
) VALUES (
  'admin', MD5('admin@omnis'), 'admin', {$q($dados['email'])}, 'Y', NULL, 'Y', NULL,
  NULL, NULL, NULL, 1, NULL, NULL, 'Admin', NULL
);

INSERT INTO `sec_users_groups` (`login`,`group_id`) VALUES ('admin', 1);

SET FOREIGN_KEY_CHECKS=1;
SQL;

        // junta e salva
        $conteudoFinal = $conteudoBase . "\n\n" . $sqlAtualizacao . "\n";
        file_put_contents($novoDumpPath, $conteudoFinal);
        Logger::log("✅ Dump atualizado salvo em: $novoDumpPath");

        // --- chama executor remoto ---
        //$executorUrl   = 'http://localhost/omnis/api/bd/index.php';
        $executorUrl   = 'https://www.omnisblueapps.com/api/BD/index.php';
        $token         = 'xwPP5ZgPV7CR3RoURIzKhVT8h35EpjMU';
        $nomeBancoNovo = 'lgpd_' . preg_replace('/\D+/', '', $dados['cnpj']);

        $payload = json_encode([
            'database' => $nomeBancoNovo,
            //'sql_file' => basename($novoDumpPath) // "Model_Modificado.sql"
            'sql_inline' => $conteudoFinal,
        ]);

        Logger::log("🚀 Log de conexão (Exec {$executorUrl}): Banco {$novoDumpPath}");

        // request HTTP
        $ch = curl_init($executorUrl);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                "Authorization: Bearer {$token}",
                'Expect:',                  // <<< desliga 100-continue
                'Connection: close'         // <<< simplifica
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HEADER => true,
            CURLOPT_TIMEOUT => 120,
        ]);
        $resp = curl_exec($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $headers = substr($resp, 0, $headerSize);
        $body = substr($resp, $headerSize);


        if ($resp === false) {
            throw new \Exception('Falha ao chamar executor: '.curl_error($ch));
        }
        curl_close($ch);

        Logger::log("📤 Payload JSON: " . $payload);
        Logger::log("🔍 JSON válido? " . (json_last_error() === JSON_ERROR_NONE ? 'Sim' : json_last_error_msg()));
        Logger::log("🚀 Executor respondeu (HTTP {$http}): {$resp}");
        Logger::log("🔄 Headers:\n{$headers}");
        Logger::log("📦 Body:\n{$body}");


        if ($http >= 200 && $http < 300) {
            Logger::log("✅ Banco criado com sucesso, agora vinculando usuário na aba do Odoo...");

            // usa o mesmo e-mail da empresa para criar o vínculo
            $this->adicionarOuAtualizarVinculoPlataformaPorEmail(
                $dados['email'],      // Empresa (pode passar email, a função resolve o ID)
                $dados['email'],      // Usuário (mesmo e-mail que foi inserido no banco)
                $dados['nome'],       // Nome que aparecerá no vínculo
                'Ativo'               // Status
            );
        }

 
        return [
            'status' => ($http >= 200 && $http < 300) ? 'ok' : 'erro',
            'dados'  => $p,
            'executor' => [
                'http' => $http,
                'body' => json_decode($resp, true)
            ]
        ];

    } catch (\Exception $e) {
        Logger::log("❌ Erro ao buscar empresa por CNPJ: " . $e->getMessage());
        return ['status' => 'erro', 'mensagem' => $e->getMessage()];
    }
}



    // private function buscarContatoPorEmailOuNome($valor) {
    //     $result = $this->odoo->searchRead('res.partner', [
    //         '|',
    //         ['email', '=', $valor],
    //         ['name', '=', $valor]
    //     ], ['id'], 0, 1);

    //     return $result[0] ?? null;
    // }


}
