<?php

use App\WebhookResponse;
use Src\Logger;

require_once __DIR__ . '/../../src/Logger.php';
require_once __DIR__ . '/Config.php';
require_once __DIR__ . '/WebhookResponse.php';
require_once __DIR__ . '/Actions/SendEmailAction.php';

//use Throwable;

class Webhook{

    private ?array $payload;
    
    public function __construct(Logger $logger, ?array $payload, ?array $headers, ?string $method) {

        $this->logger = $logger;
        $this->payload = $payload;

        $this->validateToken($headers['Authorization'] ?? null);

        $this->validateContentType($headers['Content-Type'] ?? null);
        
        $this->validateMethod($method);

        $this->validatePayload($payload);
    }

    private function response(WebhookResponse $response){

        http_response_code($response->http_code);

        $success = $response->success;
        $message = $response->message;

        $output = json_encode(compact('success', 'message'), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        //$this->logger->log  ($response, $this->payload);
        Logger::log($output);
        echo $output;
        exit;
    }

    public function errorResponse(string $message, int $http_code = 400){
        $this->response(new WebhookResponse($http_code, false, $message));
    }

    public function successResponse(string $message, int $http_code = 200){
        $this->response(new WebhookResponse($http_code, true, $message));
    }

    private function validateToken(?string $token){

        if (empty($token) || $token == null) {
            $this->errorResponse("Authorization header is missing", 401);
        }

        $token_arr = explode(" ", $token);

        if (count($token_arr) != 2 || $token_arr[0] !== 'Bearer') {
            $this->errorResponse("Authorization header must be in the format 'Bearer <token>'", 401);
        }

        $current_token = Config::get('webhook', 'token');

        if(!$current_token){
            throw new \Exception('Token not defined in configuration file');
        }

        if ($current_token != $token_arr[1]) {
            $this->errorResponse("Invalid authentication token provided", 401);
        }

        return 1;

    }

    private function validateMethod(?string $method){
        if(empty($method) || $method != "POST"){
            $this->errorResponse('This webhook only allows POST requests', 405);
        }

        return 1;
    }

    private function validateContentType(?string $contentType){
        if(empty($contentType) || $contentType == NULL || strtolower($contentType) !== 'application/json'){
            $this->errorResponse("This webhook only supports requests with 'Content-Type: application/json'", 415);
        }

        return 1;
    }

    private function validatePayload(?array $payload){
        if(empty($payload) || $payload == NULL){
            $this->errorResponse("Invalid or malformed request body");
        }

        if(!array_key_exists('action', $payload)){
            $this->errorResponse("The 'action' field is required");
        }

        return 1;
    }

    public function handle(){

        $action = $this->payload['action'];

        $actions = [
            'send-email' => SendEmailAction::class
        ];

        if (!array_key_exists($action, $actions)) {
            $this->errorResponse("Action '$action' is not supported");
        }
        
        /** @var WebhookActionInterface $handler */
        $handler = new $actions[$action];
        $handler->handle($this, $this->payload);
    }

}
